#include "TessellationPlane.h"



TessellationPlane::TessellationPlane(ID3D11Device* device, ID3D11DeviceContext* deviceContext)
{
	initBuffers(device);
}

TessellationPlane::~TessellationPlane()
{
	BaseMesh::~BaseMesh();
}

// Build shape and fill buffers.
void TessellationPlane::initBuffers(ID3D11Device* device)
{
	VertexType* vertices;
	unsigned long* indices;
	D3D11_SUBRESOURCE_DATA vertexData, indexData;

	int resolution = 50;

	//Calculate the number of vertices in the terrain mesh.
	vertexCount = 4 * resolution * resolution;
	indexCount = vertexCount;

	vertices = new VertexType[vertexCount];
	indices = new unsigned long[indexCount];

	//Calculate the increments for the X, Z and UV coordinates
	float xIncrement = 200.0f / resolution;
	float zIncrement = xIncrement;
	int currentIndex = 0;
	float u = 0.0f;
	float v = 0.0f;
	float uvIncrement = 1.0f / resolution;

	//Loop through Z, from the back to the front
	for (float z = 200.0f; z > 0.0f; z -= zIncrement)
	{
		//Loop through X, from the left to the right
		for (float x = 0.0f; x < 200.0f; x += xIncrement)
		{
			vertices[currentIndex].position = XMFLOAT3(x, 0.0f, z);
			vertices[currentIndex].normal = XMFLOAT3(0.0f, 1.0f, 0.0f);
			vertices[currentIndex].texture = XMFLOAT2(u, v);

			vertices[currentIndex + 1].position = XMFLOAT3(x, 0.0f, z - zIncrement);
			vertices[currentIndex + 1].normal = XMFLOAT3(0.0f, 1.0f, 0.0f);
			vertices[currentIndex + 1].texture = XMFLOAT2(u, v + uvIncrement);

			vertices[currentIndex + 2].position = XMFLOAT3(x + xIncrement, 0.0f, z - zIncrement);
			vertices[currentIndex + 2].normal = XMFLOAT3(0.0f, 1.0f, 0.0f);
			vertices[currentIndex + 2].texture = XMFLOAT2(u + uvIncrement, v + uvIncrement);

			vertices[currentIndex + 3].position = XMFLOAT3(x + xIncrement, 0.0f, z);
			vertices[currentIndex + 3].normal = XMFLOAT3(0.0f, 1.0f, 0.0f);
			vertices[currentIndex + 3].texture = XMFLOAT2(u + uvIncrement, v);

			indices[currentIndex] = currentIndex;
			indices[currentIndex + 1] = currentIndex + 1;
			indices[currentIndex + 2] = currentIndex + 2;
			indices[currentIndex + 3] = currentIndex + 3;

			//4 vertices drawn, increment the index and the U coordinate
			currentIndex += 4;
			u += uvIncrement;
		}

		//This row has been drawn, reset U coordinate and increment the V coordinate
		u = 0;
		v += uvIncrement;
	}

	//Set up the description of the static vertex buffer.
	D3D11_BUFFER_DESC vertexBufferDesc = { sizeof(VertexType) * vertexCount, D3D11_USAGE_DEFAULT, D3D11_BIND_VERTEX_BUFFER, 0, 0, 0 };
	vertexData = { vertices, 0 , 0 };
	device->CreateBuffer(&vertexBufferDesc, &vertexData, &vertexBuffer);

	//Set up the description of the static vertex buffer.
	D3D11_BUFFER_DESC indexBufferDesc = { sizeof(unsigned long) * indexCount, D3D11_USAGE_DEFAULT, D3D11_BIND_INDEX_BUFFER, 0, 0, 0 };
	indexData = { indices, 0, 0 };
	device->CreateBuffer(&indexBufferDesc, &indexData, &indexBuffer);

	//Release the arrays now that the buffers have been created and loaded.
	delete[] vertices;
	vertices = 0;
	delete[] indices;
	indices = 0;
}

void TessellationPlane::sendData(ID3D11DeviceContext* deviceContext, D3D_PRIMITIVE_TOPOLOGY top)
{
	unsigned int stride;
	unsigned int offset;

	// Set vertex buffer stride and offset.
	stride = sizeof(VertexType);
	offset = 0;

	deviceContext->IASetVertexBuffers(0, 1, &vertexBuffer, &stride, &offset);
	deviceContext->IASetIndexBuffer(indexBuffer, DXGI_FORMAT_R32_UINT, 0);
	deviceContext->IASetPrimitiveTopology(top);	//Set the topology to the one passed in
}